const {
  DEFAULT_LEAVE_TOTALS,
  LEAVE_TYPES,
  ANNUAL_LEAVE_DAYS_PER_MONTH,
} = require('../leaveConstants');

const MS_PER_DAY = 24 * 60 * 60 * 1000;

function createLeaveBalanceService(pool) {
  const toNumber = (value, fallback = 0) => {
    const parsed = Number(value);
    return Number.isFinite(parsed) ? parsed : fallback;
  };

  const clampYear = (value, fallback = new Date().getUTCFullYear()) => {
    const parsed = Number.parseInt(value, 10);
    if (Number.isNaN(parsed)) return fallback;
    return parsed;
  };

  const calculateInclusiveLeaveDays = (startDate, endDate) => {
    const start = new Date(startDate);
    const end = new Date(endDate);
    if (Number.isNaN(start.getTime()) || Number.isNaN(end.getTime())) return 0;
    const startUtc = Date.UTC(start.getUTCFullYear(), start.getUTCMonth(), start.getUTCDate());
    const endUtc = Date.UTC(end.getUTCFullYear(), end.getUTCMonth(), end.getUTCDate());
    if (endUtc < startUtc) return 0;
    return Math.round((endUtc - startUtc) / MS_PER_DAY) + 1;
  };

  const splitLeaveAcrossYears = (startDate, endDate) => {
    const start = new Date(startDate);
    const end = new Date(endDate);
    if (Number.isNaN(start.getTime()) || Number.isNaN(end.getTime())) return [];

    const segments = [];
    let cursor = Date.UTC(start.getUTCFullYear(), start.getUTCMonth(), start.getUTCDate());
    const endUtc = Date.UTC(end.getUTCFullYear(), end.getUTCMonth(), end.getUTCDate());
    if (endUtc < cursor) return [];

    while (cursor <= endUtc) {
      const cursorDate = new Date(cursor);
      const year = cursorDate.getUTCFullYear();
      const yearEnd = Date.UTC(year, 11, 31);
      const segmentEnd = Math.min(yearEnd, endUtc);
      const days = Math.round((segmentEnd - cursor) / MS_PER_DAY) + 1;
      segments.push({ year, days });
      cursor = segmentEnd + MS_PER_DAY;
    }

    return segments;
  };

  const calculateAnnualAllocation = (joinDate, targetYear) => {
    if (!joinDate) return DEFAULT_LEAVE_TOTALS.annual;
    const parsed = new Date(joinDate);
    if (Number.isNaN(parsed.getTime())) return DEFAULT_LEAVE_TOTALS.annual;
    const joinYear = parsed.getUTCFullYear();
    if (joinYear < targetYear) return DEFAULT_LEAVE_TOTALS.annual;
    if (joinYear > targetYear) return 0;
    const joinMonth = parsed.getUTCMonth() + 1;
    const monthsRemaining = Math.max(0, 12 - joinMonth + 1);
    return Math.min(DEFAULT_LEAVE_TOTALS.annual, monthsRemaining * ANNUAL_LEAVE_DAYS_PER_MONTH);
  };

  const fetchUserJoinDate = async (userId) => {
    const [[row]] = await pool.query(
      `SELECT s.join_date
       FROM staff s
       JOIN users u ON u.email = s.email
       WHERE u.id = ?
       LIMIT 1`,
      [userId]
    );
    return row?.join_date || null;
  };

  const seedLeaveBalancesForYear = async ({ userId, year, joinDate }) => {
    const resolvedJoinDate = joinDate || (await fetchUserJoinDate(userId));
    for (const leaveType of LEAVE_TYPES) {
      const baseAllocation =
        leaveType === 'annual'
          ? calculateAnnualAllocation(resolvedJoinDate, year)
          : DEFAULT_LEAVE_TOTALS[leaveType];

      await pool.query(
        `INSERT INTO leave_balances (user_id, year, leave_type, base_allocation, carry_over, manual_adjustment, used_days)
         VALUES (?, ?, ?, ?, 0, 0, 0)
         ON DUPLICATE KEY UPDATE base_allocation = VALUES(base_allocation)`,
        [userId, year, leaveType, baseAllocation]
      );
    }
  };

  const getLeaveBalanceRows = async (userId, year) => {
    await seedLeaveBalancesForYear({ userId, year });
    const [rows] = await pool.query('SELECT * FROM leave_balances WHERE user_id = ? AND year = ?', [userId, year]);
    return rows;
  };

  const normalizeLeaveBalanceRow = (row) => {
    if (!row) return null;
    const baseAllocation = toNumber(row.base_allocation);
    const carryOver = toNumber(row.carry_over);
    const manualAdjustment = toNumber(row.manual_adjustment);
    const usedDays = toNumber(row.used_days);
    const total = baseAllocation + carryOver + manualAdjustment;
    const remaining = Math.max(0, total - usedDays);
    return {
      id: row.id,
      userId: row.user_id,
      year: row.year,
      leaveType: row.leave_type,
      baseAllocation,
      carryOver,
      manualAdjustment,
      used: usedDays,
      total,
      remaining,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  };

  const getLeaveBalanceSummary = async (userId, year = new Date().getUTCFullYear()) => {
    const rows = await getLeaveBalanceRows(userId, year);
    const map = new Map(rows.map((row) => [row.leave_type, normalizeLeaveBalanceRow(row)]));
    const summary = {};
    for (const leaveType of LEAVE_TYPES) {
      if (!map.has(leaveType)) {
        summary[leaveType] = normalizeLeaveBalanceRow({
          id: null,
          user_id: userId,
          year,
          leave_type: leaveType,
          base_allocation: leaveType === 'annual' ? DEFAULT_LEAVE_TOTALS.annual : DEFAULT_LEAVE_TOTALS[leaveType],
          carry_over: 0,
          manual_adjustment: 0,
          used_days: 0,
          created_at: null,
          updated_at: null,
        });
        continue;
      }
      summary[leaveType] = map.get(leaveType);
    }
    return summary;
  };

  const listLeaveBalances = async ({ userId, year, leaveType } = {}) => {
    const conditions = [];
    const params = [];
    if (userId) {
      conditions.push('lb.user_id = ?');
      params.push(Number(userId));
    }
    if (year) {
      conditions.push('lb.year = ?');
      params.push(Number(year));
    }
    if (leaveType) {
      conditions.push('lb.leave_type = ?');
      params.push(leaveType);
    }
    const whereClause = conditions.length ? `WHERE ${conditions.join(' AND ')}` : '';
    const [rows] = await pool.query(
      `SELECT lb.*, u.first_name, u.last_name, u.email
       FROM leave_balances lb
       JOIN users u ON u.id = lb.user_id
       ${whereClause}
       ORDER BY lb.user_id, lb.year DESC, lb.leave_type`,
      params
    );
    return rows.map((row) => ({
      ...normalizeLeaveBalanceRow(row),
      user: {
        id: row.user_id,
        firstName: row.first_name,
        lastName: row.last_name,
        email: row.email,
      },
    }));
  };

  const updateLeaveBalanceRow = async (id, fields = {}) => {
    const allowed = {
      baseAllocation: 'base_allocation',
      carryOver: 'carry_over',
      manualAdjustment: 'manual_adjustment',
      used: 'used_days',
    };
    const sets = [];
    const params = [];

    Object.entries(fields).forEach(([key, value]) => {
      if (!(key in allowed) || value === undefined || value === null) return;
      sets.push(`${allowed[key]} = ?`);
      params.push(Number(value));
    });

    if (sets.length) {
      sets.push('updated_at = CURRENT_TIMESTAMP');
      await pool.query(`UPDATE leave_balances SET ${sets.join(', ')} WHERE id = ?`, [...params, id]);
    }

    const [rows] = await pool.query('SELECT * FROM leave_balances WHERE id = ? LIMIT 1', [id]);
    return rows.length ? normalizeLeaveBalanceRow(rows[0]) : null;
  };

  const incrementUsedDaysForLeave = async ({ userId, leaveType, startDate, endDate }) => {
    const segments = splitLeaveAcrossYears(startDate, endDate);
    if (!segments.length) return;
    for (const { year, days } of segments) {
      await seedLeaveBalancesForYear({ userId, year });
      await pool.query(
        `UPDATE leave_balances
         SET used_days = used_days + ?
         WHERE user_id = ? AND year = ? AND leave_type = ?`,
        [days, userId, year, leaveType]
      );
    }
  };

  const runAnnualAccrualForYear = async (targetYear = new Date().getUTCFullYear()) => {
    const currentYear = clampYear(targetYear);
    const previousYear = currentYear - 1;
    const [staffUsers] = await pool.query(
      `SELECT u.id AS user_id, s.join_date
       FROM users u
       JOIN staff s ON s.email = u.email`
    );

    for (const user of staffUsers) {
      const userId = Number(user.user_id);
      await seedLeaveBalancesForYear({ userId, year: currentYear, joinDate: user.join_date });
      const previousSummary = await getLeaveBalanceSummary(userId, previousYear);
      for (const leaveType of LEAVE_TYPES) {
        const carryOver = previousSummary[leaveType]?.remaining ?? 0;
        await pool.query(
          `UPDATE leave_balances
           SET carry_over = ?
           WHERE user_id = ? AND year = ? AND leave_type = ?`,
          [carryOver, userId, currentYear, leaveType]
        );
      }
    }

    return staffUsers.length;
  };

  const ensureCurrentYearBalanceForUser = async (userId, joinDate) => {
    const currentYear = new Date().getUTCFullYear();
    await seedLeaveBalancesForYear({ userId, year: currentYear, joinDate });
  };

  return {
    toNumber,
    clampYear,
    calculateInclusiveLeaveDays,
    splitLeaveAcrossYears,
    calculateAnnualAllocation,
    fetchUserJoinDate,
    seedLeaveBalancesForYear,
    getLeaveBalanceRows,
    normalizeLeaveBalanceRow,
    getLeaveBalanceSummary,
    listLeaveBalances,
    updateLeaveBalanceRow,
    incrementUsedDaysForLeave,
    runAnnualAccrualForYear,
    ensureCurrentYearBalanceForUser,
  };
}

module.exports = {
  createLeaveBalanceService,
};
