const jwt = require('jsonwebtoken');
const pool = require('../db');

const TOKEN_HEADER = 'authorization';
const BEARER_PREFIX = 'bearer ';

const ROLE_PERMISSIONS = {
  admin: ['*'],
  hr: [
    'staff:read',
    'staff:write',
    'leave:approve',
    'leave:manage',
    'attendance:settings',
    'attendance:reports',
    'announcements:manage',
    'announcements:view',
  ],
  manager: ['staff:read', 'leave:approve', 'announcements:view'],
  staff: ['announcements:view'],
};

function resolvePermissions(role = 'staff') {
  if (!role) return ROLE_PERMISSIONS.staff;
  const normalized = role.toLowerCase();
  return ROLE_PERMISSIONS[normalized] || ROLE_PERMISSIONS.staff;
}

function userHasPermission(user, permission) {
  if (!user) return false;
  if (!permission) return true;
  const permissions = user.permissions || [];
  return permissions.includes('*') || permissions.includes(permission);
}

function extractToken(req) {
  const raw = req.headers[TOKEN_HEADER];
  if (!raw || typeof raw !== 'string') return null;
  if (raw.toLowerCase().startsWith(BEARER_PREFIX)) {
    return raw.slice(BEARER_PREFIX.length).trim();
  }
  return raw.trim();
}

async function fetchUserById(userId) {
  const [rows] = await pool.query(
    'SELECT id, email, role, first_name AS firstName, last_name AS lastName FROM users WHERE id = ? LIMIT 1',
    [userId]
  );
  return rows[0] || null;
}

function requireAuth(req, res, next) {
  try {
    const token = extractToken(req);
    if (!token) {
      return res.status(401).json({ message: 'Authentication required' });
    }
    const payload = jwt.verify(token, process.env.JWT_SECRET || 'super-secret-jwt-key');
    req.user = {
      id: payload.sub || payload.id,
      email: payload.email,
      role: payload.role || 'staff',
      first_name: payload.firstName,
      last_name: payload.lastName,
      permissions: resolvePermissions(payload.role),
    };
    next();
  } catch (error) {
    console.error('[Auth] Invalid token:', error.message);
    return res.status(401).json({ message: 'Invalid or expired token' });
  }
}

function requireHrOrAdmin(req, res, next) {
  if (!req.user || !['admin', 'hr'].includes(req.user.role)) {
    return res.status(403).json({ message: 'HR or admin role required' });
  }
  next();
}

function requireRole(roles = []) {
  const normalized = Array.isArray(roles) ? roles : [roles];
  return (req, res, next) => {
    if (!req.user || !normalized.includes(req.user.role)) {
      return res.status(403).json({ message: 'Insufficient permissions' });
    }
    next();
  };
}

function requirePermission(permission) {
  return (req, res, next) => {
    if (!userHasPermission(req.user, permission)) {
      return res.status(403).json({ message: 'Insufficient permissions' });
    }
    next();
  };
}

function requireAnyPermission(permissions = []) {
  const list = Array.isArray(permissions) ? permissions : [permissions];
  return (req, res, next) => {
    if (!list.some((perm) => userHasPermission(req.user, perm))) {
      return res.status(403).json({ message: 'Insufficient permissions' });
    }
    next();
  };
}

function isPrivilegedRole(role) {
  return ['admin', 'hr'].includes(role);
}

module.exports = {
  requireAuth,
  requireHrOrAdmin,
  requireRole,
  requirePermission,
  requireAnyPermission,
  isPrivilegedRole,
  fetchUserById,
  resolvePermissions,
};
