const admin = require('firebase-admin');
const pool = require('../db');
const { createNotification } = require('../reminders');

const getAdminSdk = () => (admin.apps.length ? admin.app() : null);

async function fetchHrAndAdminUsers() {
  const [rows] = await pool.query(
    "SELECT id, email, first_name AS firstName, last_name AS lastName FROM users WHERE role IN ('admin','hr')"
  );
  return rows;
}

async function sendLeaveNotification(userId, { title, body, data, type = 'leave_update' }) {
  await createNotification(pool, getAdminSdk(), {
    userId,
    type,
    title,
    body,
    data,
  });
}

async function sendPlanNotification(userId, { title, body, data, type = 'plan_update' }) {
  await createNotification(pool, getAdminSdk(), {
    userId,
    type,
    title,
    body,
    data,
  });
}

async function sendAnnouncementNotification(userId, { title, body, data, type = 'announcement', skipPush = false }) {
  await createNotification(pool, skipPush ? null : getAdminSdk(), {
    userId,
    type,
    title,
    body,
    data,
    skipPush,
  });
}

async function broadcastAnnouncementPush(userIds, { title, body, data }) {
  if (!Array.isArray(userIds) || !userIds.length) return;
  const adminSdk = getAdminSdk();
  if (!adminSdk || !adminSdk.messaging) return;

  const [tokenRows] = await pool.query(
    'SELECT user_id, token FROM device_tokens WHERE user_id IN (?)',
    [userIds]
  );
  const tokens = tokenRows.map((row) => row.token).filter(Boolean);
  if (!tokens.length) return;

  const chunkSize = 500;
  for (let i = 0; i < tokens.length; i += chunkSize) {
    const chunk = tokens.slice(i, i + chunkSize);
    try {
      await adminSdk.messaging().sendEachForMulticast({
        tokens: chunk,
        notification: { title, body },
        data: { type: 'announcement', ...(data || {}) },
        android: {
          priority: 'high',
          notification: {
            channelId: 'announcements',
            priority: 'high',
            defaultSound: true,
            defaultVibrateTimings: true,
          },
        },
        apns: {
          payload: { aps: { sound: 'default', badge: 1 } },
        },
      });
    } catch (err) {
      console.warn('[Announcements] Broadcast push failed:', err.message);
    }
  }
}

async function sendMessageNotification(receiverId, { senderId, senderName, body, messageId }) {
  const truncatedBody = (body || '').length > 120 ? body.slice(0, 120) + '…' : (body || 'Sent an attachment');
  await createNotification(pool, getAdminSdk(), {
    userId: receiverId,
    type: 'message',
    title: senderName || 'New message',
    body: truncatedBody,
    data: {
      type: 'message',
      messageId: messageId?.toString() || '',
      senderId: senderId?.toString() || '',
    },
  });
}

async function sendGroupMessageNotification(receiverId, { senderId, senderName, groupId, groupName, body, messageId }) {
  const truncatedBody = (body || '').length > 120 ? body.slice(0, 120) + '…' : (body || 'New message');
  await createNotification(pool, getAdminSdk(), {
    userId: receiverId,
    type: 'group_message',
    title: `${groupName || 'Group'} — ${senderName || 'Someone'}`,
    body: truncatedBody,
    data: {
      type: 'group_message',
      groupId: groupId?.toString() || '',
      messageId: messageId?.toString() || '',
      senderId: senderId?.toString() || '',
    },
  });
}

async function sendGroupCallNotification(receiverId, { callerName, groupId, groupName, callId, callType }) {
  await createNotification(pool, getAdminSdk(), {
    userId: receiverId,
    type: 'group_call',
    title: `${groupName || 'Group'} — Group Call`,
    body: `${callerName || 'Someone'} started a ${callType || 'audio'} call`,
    data: {
      type: 'group_call',
      groupId: groupId?.toString() || '',
      callId: callId?.toString() || '',
    },
  });
}

module.exports = {
  fetchHrAndAdminUsers,
  sendLeaveNotification,
  sendPlanNotification,
  sendAnnouncementNotification,
  broadcastAnnouncementPush,
  sendMessageNotification,
  sendGroupMessageNotification,
  sendGroupCallNotification,
};
